<?php

namespace App\Models;

// use Illuminate\Contracts\Auth\MustVerifyEmail;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Laravel\Sanctum\HasApiTokens;
use Illuminate\Support\Facades\Crypt;

class User extends Authenticatable
{
    use HasApiTokens, HasFactory, Notifiable;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'name',
        'username',
        'email',
        'password',
        'is_admin',
        'fbr_access_token',
    ];

    /**
     * The attributes that should be hidden for serialization.
     *
     * @var array<int, string>
     */
    protected $hidden = [
        'password',
        'remember_token',
        'fbr_access_token',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'email_verified_at' => 'datetime',
        'password' => 'hashed',
        'is_admin' => 'boolean',
    ];

    /**
     * Check if user is admin
     *
     * @return bool
     */
    public function isAdmin(): bool
    {
        return $this->is_admin;
    }

    /**
     * Get the role attribute
     *
     * @return string
     */
    public function getRoleAttribute(): string
    {
        return $this->is_admin ? 'admin' : 'user';
    }

    /**
     * Set the FBR access token attribute (encrypt it)
     *
     * @param string|null $value
     */
    public function setFbrAccessTokenAttribute($value): void
    {
        $this->attributes['fbr_access_token'] = $value ? Crypt::encryptString($value) : null;
    }

    /**
     * Get the FBR access token attribute (decrypt it)
     *
     * @param string|null $value
     * @return string|null
     */
    public function getFbrAccessTokenAttribute($value): ?string
    {
        if (!$value) {
            return null;
        }

        try {
            return Crypt::decryptString($value);
        } catch (\Exception $e) {
            return null;
        }
    }

    /**
     * Get the decrypted FBR access token for display purposes
     *
     * @return string|null
     */
    public function getDecryptedFbrAccessToken(): ?string
    {
        return $this->fbr_access_token;
    }

    /**
     * Set the FBR access token directly without encryption (for already encrypted values)
     *
     * @param string|null $value
     */
    public function setEncryptedFbrAccessToken($value): void
    {
        $this->attributes['fbr_access_token'] = $value;
    }
}
